#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <QtCore/QObject>
#include <QtCore/QByteArray>
#include <qdebug.h>
#include "voicedata.h"
#include "voicesynth.h"

VoiceSynth::VoiceSynth(Model *p_model, QThread *parent) : QThread(parent) {

  int i1, i2;

  model = p_model;
  voiceData = model->currentVoiceData();
  for (i1 = 0; i1 < MAX_PARAMS; i1++) {
    shbuf[i1] = (double*)malloc(model->getNumHarmonics() * sizeof(double));
    for (i2 = 0; i2 < model->getNumHarmonics(); i2++) {
      shbuf[i1][i2] = voiceData->getScaledHarmonic(i1, i2);
    }
  }   
  a = (double *)malloc(model->getNumHarmonics() * sizeof(double));
  memset(a, 0, model->getNumHarmonics() * sizeof(double));    
  a1 = (double *)malloc(model->getNumHarmonics() * sizeof(double));
  memset(a1, 0, model->getNumHarmonics() * sizeof(double));    
  a2 = (double *)malloc(model->getNumHarmonics() * sizeof(double));
  memset(a2, 0, model->getNumHarmonics() * sizeof(double));    
  da = (double *)malloc(model->getNumHarmonics() * sizeof(double));
  memset(da, 0, model->getNumHarmonics() * sizeof(double));    
  v1 = 0;
  v2 = 0;
  v = 0;
  dv = 0;
  for (i2 = 0; i2 < 2; i2++) {
    f[i2] = (double *)malloc(model->getNumHarmonics() * sizeof(double));
    memset(f[i2], 0, model->getNumHarmonics() * sizeof(double));    
  }
  pb = 0;
  pb1 = 0;
  pb2 = 0;
  dpb = 0;
  lfo = 0;
  lfo1 = 0;
  lfo2 = 0;
  dlfo = 0;
  noise_note = 0;
  initVoiceSynth();
}

VoiceSynth::~VoiceSynth() {

  int i1, i2;

  for (i1 = 0; i1 < MAX_PARAMS; i1++) {
    free(shbuf[i1]);
  }  
  free(a);      
  free(a1);        
  free(a2);      
  free(da);  
  for (i2 = 0; i2 < 2; i2++) {
    free(f[i2]); 
  }           
}

void VoiceSynth::initVoiceSynth() {

  int i1;

  for (i1 = 0; i1 < 2; i1++) {
    dbuf[i1] = (double *)malloc(2 * MAX_FRAMES * sizeof(double));
    memset(dbuf[i1], 0, 2 * MAX_FRAMES * sizeof(double));
  }
  bufIndex = 0;
  vbuf = dbuf[bufIndex];
}  

void VoiceSynth::envStep() {

  int i2;
  voiceType *v;
  envelopeType *e;
  bool voiceActive;
  modulationEnvelope modEnv;
  modulationLfo modLfo, modSpacingLfo;
  double rnd_scale, rnd_signed, rnd_val;
  double frq_scale;

  rnd_scale = 1.0 / (double)RAND_MAX;
  frq_scale = 48000.0 / (double)model->getRate();
  v = voiceData->getVoice();
  e = voiceData->getEnvelopes();
  voiceActive = false;
  for (i2 = 0; i2 < model->getNumHarmonics(); i2++) {
    switch (e[i2].state) {
    case 1:
      if (v->samples >= shbuf[1][i2]) {
        if (e[i2].val < 1.0) {
          e[i2].val += shbuf[2][i2];
          if (e[i2].val > 1.0) {
            e[i2].val = 1.0;
            e[i2].state = 2;
          }
        } else {
          e[i2].state = 2;
        }
      }  
      voiceActive = true;
      break;
    case 2:
      if (e[i2].val > shbuf[5][i2]) {
        e[i2].val *= shbuf[3][i2];
      } else {
        e[i2].val *= shbuf[4][i2];
      }
      voiceActive = true;
      break;  
    case 3: 
      if (e[i2].val > 1e-10) {
        e[i2].val *= shbuf[6][i2];;
        voiceActive = true;
      } else {
        e[i2].val = 0;
      }
      break;  
    }
    if (e[i2].val < 1e-10) {
      e[i2].val = 0;
    }  
 
    switch (e[i2].lfo_state) {               // Env LFO
    case 0:                                  // init LFO
      e[i2].y = 0;
      e[i2].dy = 0;      
      e[i2].ddy = 0.0000008;
      e[i2].a = 0;
      e[i2].lfo_state = 1;
      break;
    case 1:                                  // rising curve, ddy > 0
      if (e[i2].y >= 0) {                    // middle point reached
        e[i2].lfo_state = 2;
        e[i2].a = 0.32 * pow(shbuf[7][i2], 0.5);
      } 
      break;
    case 2:                                        // rising curve, ddy < 0
      if ((e[i2].y >= 1.0) || (e[i2].dy <= 0)) {   // upper peak reached
        e[i2].lfo_state = 3;
        e[i2].dy = 0;
        rnd_signed = 1.0 - 2.0 * rnd_scale * rand();
        if (rnd_signed >= 0) {
          rnd_val = 1.0 + rnd_signed * pow(100.0 * shbuf[9][i2], 0.33333) ;
        } else {
          rnd_val = 1.0 + rnd_signed * 0.095 * pow(100.0 * shbuf[9][i2], 0.33333) ;
        }  
        e[i2].freqSh = shbuf[8][i2] * frq_scale; 
        e[i2].ddy = -0.0001 * rnd_val * pow(e[i2].freqSh, 2.5);
      } 
      break;
    case 3:                                  // falling curve, ddy < 0
      if (e[i2].y <0) {                      // middle point reached
        e[i2].lfo_state = 4;
        e[i2].a = 0.32 * pow(shbuf[7][i2], 0.5);
      } 
      break;
    case 4:                                       // falling curve, ddy > 0
      if ((e[i2].y < -1.0) || (e[i2].dy > 0)) {   // lower peak reached
        e[i2].lfo_state = 1;
        e[i2].dy = 0;
        rnd_signed = 1.0 - 2.0 * rnd_scale * rand();
        if (rnd_signed >= 0) {
          rnd_val = 1.0 + rnd_signed * pow(100.0 * shbuf[9][i2], 0.33333) ;
        } else {
          rnd_val = 1.0 + rnd_signed * 0.095 * pow(100.0 * shbuf[9][i2], 0.33333) ;
        }  
        e[i2].freqSh = shbuf[8][i2] * frq_scale; 
        e[i2].ddy = 0.0001 * rnd_val * pow(e[i2].freqSh, 2.5);
      } 
      break;
    }
    if (e[i2].freqSh != shbuf[8][i2] * frq_scale) {     // Update ddy if freq parameter has changed
      e[i2].freqSh = shbuf[8][i2] * frq_scale;
      rnd_signed = 1.0 - 2.0 * rnd_scale * rand();
      if (rnd_signed >= 0) {
        rnd_val = 1.0 + rnd_signed * pow(100.0 * shbuf[9][i2], 0.33333) ;
      } else {
        rnd_val = 1.0 + rnd_signed * 0.095 * pow(100.0 * shbuf[9][i2], 0.33333) ;
      }  
      if ((e[i2].lfo_state == 1) || (e[i2].lfo_state == 4)) {
        e[i2].ddy = 0.0001 * rnd_val * pow(e[i2].freqSh, 2.5);
      } else {
        e[i2].ddy = -0.0001 * rnd_val * pow(e[i2].freqSh, 2.5);
      }   
    }
    e[i2].dy += e[i2].ddy;
    e[i2].y += e[i2].dy;
    e[i2].lfo = e[i2].a * e[i2].y;
  }
  for (i2 = 0; i2 < MAX_SOURCES; i2++) {
    switch (v->filterMixEnvState[i2]) {
    case 1:
      if (v->samples - v->filterMixSampleStack[i2] >= voiceData->getFilterMixDelay(i2)) {
        v->filterMixEnvState[i2] = 2;
      }
      break;
    case 2: 
      v->filterMixEnv[i2] += 1.0 / (10.0 + voiceData->getFilterMixAttack(i2));
      if (v->filterMixEnv[i2] > 1.0) {
        v->filterMixEnv[i2] = 1.0;
        if (voiceData->getFilterMixHold(i2)) {
          v->filterMixEnvState[i2] = 3;
          v->filterMixSampleStack[i2] = v->samples;
        } else {
          v->filterMixEnvState[i2] = 4;
      }
    }
    break;     
    case 3: 
      if (v->samples - v->filterMixSampleStack[i2] >= voiceData->getFilterMixDelay(i2)) {
        v->filterMixEnvState[i2] = 4;
      }
      break;     
    case 4: 
      v->filterMixEnv[i2] -= 1.0 / (10.0 + voiceData->getFilterMixDecay(i2));
      if (v->filterMixEnv[i2] < 0) {
        v->filterMixEnv[i2] = 0;
        if (voiceData->getFilterMixLfo(i2)) {
          if (voiceData->getFilterMixHold(i2)) {
            v->filterMixEnvState[i2] = 1;
            v->filterMixSampleStack[i2] = v->samples;
          } else {
            v->filterMixEnvState[i2] = 2;
          }
        } else {
          v->filterMixEnvState[i2] = 0;
        }  
      }
      break;     
    }
  }
  for (i2 = 0; i2 < 2; i2++) {
    modEnv = voiceData->getFilterModEnvelope(i2);
    modLfo = voiceData->getFilterModLfo(i2);
    modSpacingLfo = voiceData->getFilterModSpacingLfo(i2);
    switch (v->filterModEnvState[i2]) {
    case 1:
      if (v->samples >= modEnv.delay) {
        if (v->filterModEnv[i2] < 1.0) {
          v->filterModEnv[i2] += modEnv.attack;
          if (v->filterModEnv[i2] > 1.0) {
            v->filterModEnv[i2] = 1.0;
            v->filterModEnvState[i2] = 2;
          }
        } else {
          v->filterModEnvState[i2] = 2;
        }
      }  
      break;
    case 2:
      if (v->filterModEnv[i2] > modEnv.level) {
        v->filterModEnv[i2] *= modEnv.decay1;
      } else {
        v->filterModEnv[i2] *= modEnv.decay2;
      }  
      break;  
    case 3: 
      if (v->filterModEnv[i2] > 1e-20) {
        v->filterModEnv[i2] *= modEnv.release;
      } else {
        v->filterModEnv[i2] = 0;
      }
      break;  
    }
    if (v->filterModEnv[i2] < 1e-20) {
      v->filterModEnv[i2] = 0;
    }
    switch (v->filterModLfoState[i2]) {
    case 1:
      if (v->samples >= modLfo.delay) {
        if (v->filterModLfo[i2] < 1.0) {
          v->filterModLfo[i2] += modLfo.freq;
          if (v->filterModLfo[i2] > 1.0) {
            v->filterModLfo[i2] = 1.0;
            v->filterModLfoState[i2] = 2;
          }
        } else { 
          v->filterModLfoState[i2] = 2;
        }
      }  
      break;
    case 2:
      if (v->filterModLfo[i2] > 0) {
        v->filterModLfo[i2] -= modLfo.freq;
        if (v->filterModLfo[i2] < 0) {
          v->filterModLfo[i2] = 0;
          v->filterModLfoState[i2] = 1;
        }
      } else {
        v->filterModLfoState[i2] = 1;
      }
      break;  
    }
    switch (v->filterModSpacingLfoState[i2]) {
    case 1:
      if (v->samples >= modSpacingLfo.delay) {
        if (v->filterModSpacingLfo[i2] < 1.0) {
          v->filterModSpacingLfo[i2] += modSpacingLfo.freq;
          if (v->filterModSpacingLfo[i2] > 1.0) {
            v->filterModSpacingLfo[i2] = 1.0;
            v->filterModSpacingLfoState[i2] = 2;
          }
        } else { 
          v->filterModSpacingLfoState[i2] = 2;
        }
      }  
      break;
    case 2:
      if (v->filterModSpacingLfo[i2] > 0) {
        v->filterModSpacingLfo[i2] -= modSpacingLfo.freq;
        if (v->filterModSpacingLfo[i2] < 0) {
          v->filterModSpacingLfo[i2] = 0;
          v->filterModSpacingLfoState[i2] = 1;
        }
      } else {
        v->filterModSpacingLfoState[i2] = 1;
      }
      break;  
    }
  }
  v->active = voiceActive;
  v->samples += STEP;
}

void VoiceSynth::filterStep(int filterIndex, int harmonicIndex) {

  int i2;
  voiceType *v;
  double base_frq, h_frq, x1, x2, cutoff, spacing;
  int f_index;
  modulationEnvelope modEnv;
  modulationLfo modLfo, modSpacingLfo;
  double shq[32];
  double offsetOsc[2], offsetNoise;
  double fs0, f3;

  v = voiceData->getVoice();
  modelFilter[filterIndex] = voiceData->getFilter(filterIndex);
  modEnv = voiceData->getFilterModEnvelope(filterIndex);
  modLfo = voiceData->getFilterModLfo(filterIndex);
  modSpacingLfo = voiceData->getFilterModSpacingLfo(filterIndex);
  for (i2 = 0; i2 < 32; i2++) {
    shq[i2] = 1.0 / (double)(i2 + 1);
  }
  base_frq = freqConst * exp((model->getScalaNote(v->note) + pb + lfo)*log2div12);
  
  for (i2 = 0; i2 < 2; i2++) {
    offsetOsc[i2] = freqConst * exp(voiceData->getDelta(2 * i2)*log2div12);
  }
  offsetNoise = freqConst * exp(voiceData->getDelta(4)*log2div12);
  for (i2 = model->getHarmonicSplit(harmonicIndex); i2 <  model->getHarmonicSplit(harmonicIndex) + voiceData->getHMax(harmonicIndex, 0); i2++) {
    h_frq = 1.0;
    if ((harmonicIndex % 4) == 0) { // Osc 1
      h_frq = (i2 - model->getHarmonicSplit(harmonicIndex) + 1) * base_frq;
    } else if ((harmonicIndex % 4) == 1) { // Osc 2
      h_frq = (voiceData->getDelta(1) * (double)(i2 - model->getHarmonicSplit(harmonicIndex)) + 1.0) * base_frq * offsetOsc[0];      
    } else if ((harmonicIndex % 4) == 2) { // Osc 3   
      f3 = base_frq * offsetOsc[1];
      fs0 = (voiceData->getDelta(3) < 0) ? 400.0 * voiceData->getDelta(3) : 4000.0 * voiceData->getDelta(3);
      switch (voiceData->getOsc3Mode()) {
        case 0:
          h_frq = fs0 + f3 * (i2 - model->getHarmonicSplit(harmonicIndex) + 1);
          break; 
        case 1:
          h_frq = fs0 + f3 * 8.0 * shq[i2 - model->getHarmonicSplit(harmonicIndex)];
          break;
        case 2:
          h_frq = ((i2 - model->getHarmonicSplit(harmonicIndex)) < 16)
                ? fs0 + f3 * 8.0 * shq[i2 - model->getHarmonicSplit(harmonicIndex)]
                : fs0 + f3 * (i2 - model->getHarmonicSplit(harmonicIndex) + 1);
          break;
        case 3:
          h_frq = ((i2 - model->getHarmonicSplit(harmonicIndex)) < 8)
                ? fs0 + f3 * 8.0 * shq[i2 - model->getHarmonicSplit(harmonicIndex)]        
                : fs0 + f3 * (i2 - model->getHarmonicSplit(harmonicIndex) + 1);
          break;
        }
    } else if ((harmonicIndex % 4) == 3) { // Noise
      if (voiceData->getNoiseSubHarmonicMode()) {
        h_frq = 8.0 * shq[i2 - model->getHarmonicSplit(harmonicIndex)] * (voiceData->getDelta(5) * (double)(i2 - model->getHarmonicSplit(harmonicIndex)) + 1.0) * base_frq * offsetNoise;
      } else {
        h_frq = (voiceData->getDelta(5) * (double)(i2 - model->getHarmonicSplit(harmonicIndex)) + 1.0) * base_frq * offsetNoise;
      }
    }
      
    cutoff = voiceData->getFilterCutoff(filterIndex) + modEnv.amount * v->filterModEnv[filterIndex] + modLfo.depth * v->filterModLfo[filterIndex];
    spacing = voiceData->getFilterSpacing(filterIndex) + 6.0 * modSpacingLfo.depth * v->filterModSpacingLfo[filterIndex];
    x1 = (log(h_frq / freqConst) / log2div12 - (cutoff + (model->getScalaNote(v->note) + pb + lfo) * voiceData->getFilterKeyTracking(filterIndex))) / spacing;
    f_index = (int)floor(x1);
    x2 = x1 - f_index;

    if (f_index < 0) {
      f[filterIndex][i2] = 1.0 + voiceData->getFilterDepth(filterIndex) * (1e-5 * pow(modelFilter[filterIndex][0], 2.5) - 1.0);
    } else if (f_index >= FILTER_LEN - 1) {
      f[filterIndex][i2] = 1.0 + voiceData->getFilterDepth(filterIndex) * (1e-5 * pow(modelFilter[filterIndex][FILTER_LEN - 1], 2.5) - 1.0);
    } else {
      f[filterIndex][i2] = 1.0 + voiceData->getFilterDepth(filterIndex) * (1e-5 * pow(((1.0 - x2) * modelFilter[filterIndex][f_index] 
                                                            + x2 * modelFilter[filterIndex][f_index + 1]), 2.5) - 1.0); 
    }

    x1 = voiceData->getFilterMixMorph(harmonicIndex % 4) + voiceData->getFilterMixEnvAmount(harmonicIndex % 4) * v->filterMixEnv[harmonicIndex % 4];
    if (x1 < 0) x1 = 0;
    if (x1 > 1) x1 = 1;
    f[filterIndex][i2] *= (double)filterIndex + (double)(1 - 2 * filterIndex) * (1.0 - x1);
  }
}

void VoiceSynth::run() {

  double phi[MAX_OSC], dphi[MAX_OSC];
  int i1, i2, i3, i5, i6, i7, i8, maxHarmonic[MAX_SOURCES], o[8];
  double frq[MAX_OSC], w[MAX_OSC], frame, cframe, x1, x2, sd, cd, eo[8];
  double *s, *c;
  voiceType *voice;
  envelopeType *env;
  double rscale, r, xbuf[8][3], ybuf[8][3], alpha[8], cw0[8], w0, fsw0, w3, bw, fb, sp;
  double c_a, c_b, c_c, c_d, c_x0, c_x1, c_x2, c_max, c_aa;
  int maxCounter;
  double tempMaxLevel, maxLevel, vc;
  double panCenter, panWidth, panP, panQ, pan[2];
  int panMode;
  double shq, sh_phi[SUBHARMONIC_LEN], sh_dphi[SUBHARMONIC_LEN];
  double envFreqDiffusor[SUBHARMONIC_LEN], noiseEnvFreqDiffusor;
  double detune[MAX_OSC], detuneConst;
  double c_pi;
  double volumeTrackingAndVelocity, c63;
  int trackingNote;
  
  c_pi = 100.0 * M_PI;
  c63 = 1.0 / 63.0;
  i1 = 0;
  rscale = 4.0 / (double)RAND_MAX;
  s = (double *)malloc((1 + model->getHarmonicSplit(1)) * sizeof(double));
  memset(s, 0, (1 + model->getHarmonicSplit(1)) * sizeof(double));
  c = (double *)malloc((1 + model->getHarmonicSplit(1)) * sizeof(double));
  memset(c, 0, (1 + model->getHarmonicSplit(1)) * sizeof(double));
  env = voiceData->getEnvelopes();
  for (i2 = 0; i2 < MAX_OSC; i2++) {
    phi[i2] = 0;
    detune[i2] = 0;
    w[i2] = 0;
  }  
  for (i2 = 0; i2 < SUBHARMONIC_LEN; i2++) {
    sh_phi[i2] = 0;
    sh_dphi[i2] = 0;
  }
  for (i2 = 0; i2 < 3; i2++) {
    for (i3 = 0; i3 < 8; i3++) {
      xbuf[i3][i2] = 0;
      ybuf[i3][i2] = 0;
    }  
  }  
  voice = voiceData->getVoice();
  pb1 = 0;
  pb2 = 0;
  dpb = 0;
  pb = 0;
  maxCounter = 0;
  tempMaxLevel = 0;
  maxLevel = 0;
  maxHarmonic[0] = model->getHarmonicSplit(1);
  while(model->doSynthesis()) {
    if (!noise_note) noise_note = voice->note;
    if (noise_note < voice->note) {
      if (noise_note - voice->note < model->getMaxNoiseStep()) {
        noise_note += model->getMaxNoiseStep();
      } else {
        noise_note = voice->note;
      }
    }
    if (noise_note > voice->note) {
      if (noise_note - voice->note > model->getMaxNoiseStep()) {
        noise_note -= model->getMaxNoiseStep();
      } else {
        noise_note = voice->note;
      }
    }
    for (i1 = 0; i1 < MAX_PARAMS; i1++) {
      for (i2 = 0; i2 < model->getNumHarmonics(); i2++) {
        shbuf[i1][i2] = voiceData->getScaledHarmonic(i1, i2);
      }
    }
    voiceData->getPan(panCenter, panWidth, panMode);
    panP = 0;
    switch (panMode) {
    case 0:
      panP = panCenter - panWidth * ((double)voice->note - 60.0) / 15.0;
      break;
    case 1:
      panP = panCenter + panWidth * ((double)voice->note - 60.0) / 15.0;
      break;
    case 2:
      if (voice->note % 2) {
        panP = panCenter + panWidth * ((double)voice->note - 60.0) / 15.0;
      } else {
        panP = panCenter - panWidth * ((double)voice->note - 60.0) / 15.0;
      }
      break;
    case 3:
      if (voice->note % 2) {
        panP = panCenter - panWidth * ((double)voice->note - 60.0) / 15.0;
      } else {
        panP = panCenter + panWidth * ((double)voice->note - 60.0) / 15.0;
      }
      break;
    }
    if (panP < -1.0) panP = -1.0;
    else if (panP > 1.0) panP = 1.0;
    panQ = 0.09566214 * (1.0 - panP * panP);
    pan[0] = 0.5 + 0.5 * panP + panQ;
    pan[1] = 0.5 - 0.5 * panP + panQ;
    voiceData->getCompressorCoeff(c_x0, c_x1, c_a, c_b, c_c, c_d, c_aa);
    c_x2 = c_a * c_x1 + c_b * c_x1 * c_x1 + c_c;
    c_max = (double)MAX_LEVEL * (1.0 / sqrt((double)model->getPoly())) * 0.95;
    memset(dbuf[bufIndex], 0, 2 * model->getBufSize() * sizeof(double));
    for (i2 = 0; i2 < model->getBufSize() / STEP; i2++) {
      if (voice->active) {
        for (i5 = 0; i5 < model->getNumHarmonics(); i5++) {
          a[i5] = a1[i5];
          da[i5] = (a2[i5] - a1[i5]) / (double)STEP;
        }
        for (i3 = 0; i3 < MAX_OSC; i3++) {
          detune[i3] = 0;
          for (i5 = model->getHarmonicSplit(i3); i5 < model->getHarmonicSplit(i3 + 1); i5++) {
            detune[i3] += env[i5].lfo;
          }
          detuneConst = (i3) ? 0.04 : 0.02;
          if (i3 == 3) {
            detuneConst = 0.16;
          }
          detune[i3] *= detuneConst * voiceData->getDetune();
          detune[i3] += voiceData->getCurrentVoiceDetune();
        }
        bw = voiceData->getNoiseBandwidth();
        frq[0] = freqConst * exp((model->getScalaNote(voice->note) + pb2 + lfo2 + detune[0])*log2div12);
        frq[1] = freqConst * exp((model->getScalaNote(voice->note) + pb2 + lfo2 + detune[1] + voiceData->getDelta(0))*log2div12);
        frq[2] = freqConst * exp((model->getScalaNote(voice->note) + pb2 + lfo2 + detune[2] + voiceData->getDelta(2))*log2div12);
        frq[3] = freqConst * exp((model->getScalaNote(voice->note) + pb2 + lfo2 + detune[3])*log2div12);
        frq[4] = freqConst * exp((model->getScalaNote(voice->note) + pb2 + lfo2 + detune[4])*log2div12);
        fb = freqConst * exp((model->getScalaNote(noise_note) + voiceData->getDelta(4) + pb2 + lfo2 + detune[0])*log2div12);
        maxHarmonic[0] = (int)(voice->cut / frq[0]);
        maxHarmonic[1] = (int)(voice->cut / (frq[1] * voiceData->getDelta(1)));
        maxHarmonic[2] = SUBHARMONIC_LEN;
        sp = fb * voiceData->getDelta(5);
        maxHarmonic[3] = (int)((voice->cut - fb) / sp);
        for (i5 = 0; i5 < MAX_SOURCES; i5++) {
          if (maxHarmonic[i5] > voiceData->getMaxHarmonic(i5)) {
            maxHarmonic[i5] = voiceData->getMaxHarmonic(i5);
          }
        }
        for (i5 = 0; i5 < maxHarmonic[3]; i5++) {
          noiseEnvFreqDiffusor = 0.0002 * (double)voiceData->getNoiseFreqDiffusion() * env[model->getHarmonicSplit(3) + i5].lfo;
          shq = 8.0 / ((double)(i5 + 1) + noiseEnvFreqDiffusor);
          if (voiceData->getNoiseSubHarmonicMode()) {
            w0 = 2.0 * M_PI * (fb + shq * (sp + fb * noiseEnvFreqDiffusor)) / (double)model->getRate();
          } else {
            w0 = 2.0 * M_PI * (fb + (double)i5 * (sp + fb * noiseEnvFreqDiffusor)) / (double)model->getRate();
          }
          alpha[i5] = sin(w0)*sinh(log(2)/2 * bw * w0/sin(w0));
          cw0[i5] = cos(w0);
        }                                  
        pb = pb1;
        lfo = lfo1;
        frq[0] = freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[0])*log2div12);
        frq[1] = freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[1] + voiceData->getDelta(0))*log2div12);
        frq[2] = freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[2] + voiceData->getDelta(2))*log2div12);
        frq[3] = freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[3])*log2div12);
        frq[4] = freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[4])*log2div12);
        for (i3 = 0; i3 < MAX_OSC; i3++) {
          w[i3] = 2.0 * M_PI * frq[i3];
        }
        w3 = 2.0 * M_PI * freqConst * exp((model->getScalaNote(voice->note) + pb + lfo + detune[2] + voiceData->getDelta(2))*log2div12);
        fsw0 = (voiceData->getDelta(3) < 0) ? 2.0 * M_PI * 400.0 * voiceData->getDelta(3) / model->getRate() : 2.0 * M_PI * 4000.0 * voiceData->getDelta(3) / model->getRate(); // Frequency Shifter, hier _kein_ Spacing!
        dphi[0] = w[0] / (double)model->getRate();
        dphi[1] = (w[1] / (double)model->getRate());
        dphi[2] = (w[1] / (double)model->getRate()) * voiceData->getDelta(1);
        dphi[3] = dphi[0]; // TODO Obsolet ?
        dphi[4] = dphi[0] * voiceData->getDelta(3); // TODO Obsolet ?
        for (i7 = 0; i7 < maxHarmonic[2]; i7++) {
          while (sh_phi[i7] < -c_pi) {
            sh_phi[i7] += c_pi;
          }
          while (sh_phi[i7] > c_pi) {
            sh_phi[i7] -= c_pi;
          }
          for (i5 = 0; i5 < maxHarmonic[2]; i5++) {
            envFreqDiffusor[i5] = 0.0001 * (double)voiceData->getOsc3FreqDiffusion() * env[model->getHarmonicSplit(2) + i5].lfo;
          }
          switch (voiceData->getOsc3Mode()) {
          case 0:
            sh_dphi[i7] = fsw0 + w3 * (envFreqDiffusor[i7] + (double)(i7 + 1)) / model->getRate();
            if (sh_dphi[i7] * model->getRate() > (2.0 * M_PI) * voice->cut) {
              maxHarmonic[2] = i7;
            }
            break;
          case 1:
            sh_dphi[i7] = fsw0 + w3 * 8.0 / (double)((i7 + 1) * model->getRate());
            break;
          case 2:
            sh_dphi[i7] = (i7 < 16) ? fsw0 + w3 * 4.0 / (double)((i7 + 1) * model->getRate())
                                    : fsw0 + w3 * 2.0 * (double)(i7 - 14) / model->getRate();
            if ((i7 >=16) && (sh_dphi[i7] * model->getRate() > (2.0 * M_PI) * voice->cut)) {
              maxHarmonic[2] = i7;
            }
            break;
          case 3:
            sh_dphi[i7] = (i7 < 8) ? fsw0 + w3 * 2.0 / (double)((i7 + 1) * model->getRate())
                                    : fsw0 + w3 * (double)(i7 - 14) / model->getRate();
            if ((i7 >=8) && (sh_dphi[i7] * model->getRate() > (2.0 * M_PI) * voice->cut)) {
              maxHarmonic[2] = i7;
            }
            break;
          }
        }
        for (i6 = 0; i6 < STEP; i6++) {          
          for (i7 = 0; i7 < MAX_OSC; i7++) {
            while (phi[i7] < -c_pi) phi[i7] += c_pi;
            while (phi[i7] > c_pi) phi[i7] -= c_pi;
          }
          frame = 0;
          for (i7 = 0; i7 < 8; i7++) {
            o[i7] = model->getHarmonicSplit(i7 + 1);
          }  
           x2 = voiceData->getMorphing();
            x1 = 1.0 - x2;
            if (maxHarmonic[0]) {
              s[0] = sin(phi[0]);
              c[0] = cos(phi[0]);
            }
            for (i3 = 0; i3 < maxHarmonic[0]; i3++) {
              s[i3+1] = s[i3] * c[0] + s[0] * c[i3];
              c[i3+1] = c[i3] * c[0] - s[0] * s[i3];
              frame += s[i3] * (x1 * a[i3] + x2 * a[i3 + o[3]]);  
              a[i3] += da[i3];
              a[i3 + o[3]] += da[i3 + o[3]];
            }
            if (maxHarmonic[1]) {
              s[0] = sin(phi[1]);
              c[0] = cos(phi[1]);
              sd = sin(phi[2]);
              cd = cos(phi[2]);
            }  
            for (i3 = 0; i3 < maxHarmonic[1]; i3++) {
              s[i3+1] = s[i3] * cd + sd * c[i3];
              c[i3+1] = c[i3] * cd - sd * s[i3];
              frame += s[i3] * (x1 * a[i3 + o[0]] + x2 * a[i3 + o[4]]); 
              a[i3 + o[0]] += da[i3 + o[0]];
              a[i3 + o[4]] += da[i3 + o[4]];
            }
            for (i3 = 0; i3 < maxHarmonic[2]; i3++) { //SubHarmonic
              frame += sin(sh_phi[i3]) * (x1 * a[i3 + o[1]] + x2 * a[i3 + o[5]]);
              a[i3 + o[1]] += da[i3 + o[1]];
              a[i3 + o[5]] += da[i3 + o[5]];
            }
            for (i3 = 0; i3 < maxHarmonic[3]; i3++) { // Noise
              r = rand() * rscale * (x1 * a[i3 + o[2]] + x2 * a[i3 + o[6]]); // Amplitude Envelope * Input of Bandpass
              xbuf[i3][2] = xbuf[i3][1];
              xbuf[i3][1] = xbuf[i3][0];
              xbuf[i3][0] = r;
              ybuf[i3][2] = ybuf[i3][1];
              ybuf[i3][1] = ybuf[i3][0];
              ybuf[i3][0] = (alpha[i3]/(1.0+alpha[i3])) * xbuf[i3][0] + (-alpha[i3]/(1.0+alpha[i3])) * xbuf[i3][2]
                              + (2.0*cw0[i3]/(1.0+alpha[i3])) * ybuf[i3][1] - ((1.0-alpha[i3])/(1.0+alpha[i3])) * ybuf[i3][2];   
              frame += 30.0 * ybuf[i3][0]; 
              a[i3 + o[2]] += da[i3 + o[2]];
              a[i3 + o[6]] += da[i3 + o[6]];
            }
          for (i3 = maxHarmonic[0]; i3 < model->getHarmonicSplit(1); i3++) {
            a[i3] += da[i3];
            a[i3 + o[3]] += da[i3 + o[3]];
          }
          for (i3 = maxHarmonic[1]; i3 < model->getHarmonicSplit(2) - model->getHarmonicSplit(1); i3++) {
            a[i3 + o[0]] += da[i3 + o[0]];
            a[i3 + o[4]] += da[i3 + o[4]];
          }
          for (i3 = maxHarmonic[2]; i3 < model->getHarmonicSplit(3) - model->getHarmonicSplit(2); i3++) {
            a[i3 + o[1]] += da[i3 + o[1]];
            a[i3 + o[5]] += da[i3 + o[5]];
          }
          for (i3 = maxHarmonic[3]; i3 < model->getHarmonicSplit(4) - model->getHarmonicSplit(3); i3++) {
            a[i3 + o[2]] += da[i3 + o[2]];
            a[i3 + o[6]] += da[i3 + o[6]];
          }
          if (fabs(frame) > tempMaxLevel) {
            tempMaxLevel = fabs(frame);
          } // TODO Volume Tracking * Velocity Depth * frame
          if (fabs(frame) <= c_x0) {
            cframe = frame;
          } else if (fabs(frame) < c_x1) {
            if (frame > 0) {
              cframe = c_a * frame + c_b * frame * frame + c_c;
            } else {
              cframe = c_a * frame - c_b * frame * frame - c_c;
            }  
          } else {
            if (frame > 0) {
              cframe = c_x2 + c_d * (frame - c_x2);
            } else { 
              cframe = -c_x2 + c_d * (frame + c_x2);
            }  
          }
          if (fabs(cframe > c_max)) {
            if (cframe > 0) {
              cframe = c_max + 0.0002 * (cframe - c_max);
            } else { 
              cframe = -c_max + 0.0002 * (cframe + c_max);
            }  
          }
          if (maxCounter == 0) {
            maxLevel = tempMaxLevel;
            tempMaxLevel = 0;
            if (maxLevel > c_x1) {
              vc = 1250.0 * c_aa;
              if (vc < voice->cut) {
                voice->cut = vc;
              }
            } else if (maxLevel > c_x0) {
              vc = 20000.0 - (maxLevel - c_x0) * (20000.0 - 1250.0 * c_aa)/(c_x1 - c_x0);
              if (vc < voice->cut) {
                voice->cut = vc;
              }
            }
            maxCounter = (int)(512.0 + 128.0 * (double)rand() * rscale);
          } else {
            maxCounter--;
          }
          dbuf[bufIndex][2 * STEP * i2 + 2 * i6] = pan[0] * cframe;
          dbuf[bufIndex][2 * STEP * i2 + 2 * i6 + 1] = pan[1] * cframe;
          pb += dpb;
          for (i7 = 0; i7 < MAX_OSC; i7++) {
            phi[i7] += dphi[i7];
          }
          for (i7 = 0; i7 < maxHarmonic[2]; i7++) {
            sh_phi[i7] += sh_dphi[i7];
          }
          v += dv;  
        }  
      }
      for (i5 = 0; i5 < model->getNumHarmonics(); i5++) {
        a1[i5] = a2[i5];
      }
      v1 = v2;
      v2 = voice->velocity;
      dv = (v2 - v1) / (double)STEP;
      v = v1;  
      pb1 = pb2;
      pb2 = 2.0 * voiceData->getPitchBend();
      dpb = (pb2 - pb1) / (double)STEP;
      lfo1 = lfo2;
      lfo2 = voiceData->getLFO();
      dlfo = (lfo2 - lfo1) / (double)STEP;
      envStep();
      for (i8 = 0; i8 < 2; i8++) {
        for (i7 = 0; i7 < MAX_SOURCES * MAX_SCENES; i7++) {
          filterStep(i8, i7);
        }  
      }  
      voiceData->lfoStep();
      for (i7 = 0; i7 < 4; i7++) {
        eo[i7] = voiceData->getEvenOdd(i7);
        eo[i7 + 4] = voiceData->getEvenOdd(i7);
      }
      trackingNote = voice->note;
      if (trackingNote > 126) trackingNote = 126;
      volumeTrackingAndVelocity = pow((2.0 * voiceData->getVolumeTracking() - 1.0) * c63 * (double)(trackingNote - 63) + 1.0, 3)
                                * pow((1.0 - voiceData->getVelocityDepth()) + voiceData->getVelocityDepth() * (double)voice->velocity / 127.0, 2);
      for (i7 = 0; i7 < MAX_SOURCES * MAX_SCENES; i7++) {
        x2 = voiceData->getFilterMixMode(i7 % 4);
        x1 = 1.0 - x2;
        for (i5 = model->getHarmonicSplit(i7); i5 < model->getHarmonicSplit(i7) + voiceData->getHMax(i7, 0); i5++) {
          if (i5 % 2 == 0) {
            a2[i5] = volumeTrackingAndVelocity * eo[i7] * voiceData->getMasterVolume() * env[i5].val * (1.0 + env[i5].lfo) * shbuf[0][i5] 
                       * (x1 * 32.0 * f[0][i5] * f[1][i5] + x2 * (f[0][i5] + f[1][i5]));
          } else {
            a2[i5] = volumeTrackingAndVelocity * (1.0 - eo[i7]) * voiceData->getMasterVolume() * env[i5].val * (1.0 + env[i5].lfo) * shbuf[0][i5] 
                       * (x1 * 32.0 * f[0][i5] * f[1][i5] + x2 * (f[0][i5] + f[1][i5]));
          }
        }
        for (i5 = model->getHarmonicSplit(i7) + voiceData->getHMax(i7, 0); 
             i5 < model->getHarmonicSplit(i7) + voiceData->getMaxHarmonic(i7 % 4); i5++) {
          a2[i5] = 0;
        }
      }
    }
    model->voiceMutex.lock();
    vbuf = dbuf[bufIndex];
    bufIndex = !bufIndex; 
    model->voiceReadyCounter++;
    if (model->voiceReadyCounter == model->getPoly()) {
      model->voiceReadyCounter = 0;
      model->synthWait.wakeOne();
    }
    model->voiceWait.wait(&model->voiceMutex);
    model->voiceMutex.unlock();
  }
  free(s);
  free(c);
}

void VoiceSynth::loadVoiceData(int index) {

  voiceData = model->getVoiceData(index);
}
