// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sys/stat.h>
#include <dirent.h>
#include <cerrno>
#include <sys/wait.h>
#include <spawn.h>
#include <signal.h>

#include "../canlxx.h"
#include "../opensslutil.h"
#include "../fileutil.h"

//The following test will cover the NIST Public Key Interoperability
//Test Suite (PKITS) Certificate PAth Validation
//See: http://csrc.nist.gov/groups/ST/crypto_apps_infra/documents/PKITS.pdf

static std::string BASE_LOCATION = "./NIST_PKITSTest_data/";

 std::string cert_file_names[] = {
//"4.1 Signature Verification"
   "Valid Certificate Path Test1",
   "Invalid CA Signature Test2", 
   "Invalid EE Signature Test3",
   "Valid DSA Signatures Test4",
   "Valid DSA Parameter Inheritance Test5", 
   "Invalid DSA Signature Test6",
//"4.2 Validity Periods"
   "Invalid CA notBefore Date Test1",
   "Invalid EE notBefore Date Test2",
   "Valid pre2000 UTC notBefore Date Test3",
   "Valid GeneralizedTime notBefore Date Test4",
   "Invalid CA notAfter Date Test5",
   "Invalid EE notAfter Date Test6",
   "Invalid pre2000 UTC EE notAfter Date Test7",
   "Valid GeneralizedTime notAfter Date Test8",
//"4.3 Verifying Name Chaining"
   "Invalid Name Chaining EE Test1",
   "Invalid Name Chaining Order Test2",
   "Valid Name Chaining Whitespace Test3",
   "Valid Name Chaining Whitespace Test4",
   "Valid Name Chaining Capitalization Test5",
   "Valid Name Chaining UIDs Test6",
   "Valid RFC3280 Mandatory Attribute Types Test7",
   "Valid RFC3280 Optional Attribute Types Test8",
   "Valid UTF8String Encoded Names Test9",
   "Valid Rollover from PrintableString to UTF8String Test10",
   "Valid UTF8String Case Insensitive Match Test11",
//"4.4 Basic Certificate Revocation Tests"
   "Missing CRL Test1",
   "Invalid Revoked CA Test2",
   "Invalid Revoked EE Test3",
   "Invalid Bad CRL Signature Test4",
   "Invalid Bad CRL Issuer Name Test5",
   "Invalid Wrong CRL Test6",
   "Valid Two CRLs Test7",
   "Invalid Unknown CRL Entry Extension Test8",
   "Invalid Unknown CRL Extension Test9",
   "Invalid Unknown CRL Extension Test10",
   "Invalid Old CRL nextUpdate Test11",
   "Invalid pre2000 CRL nextUpdate Test12",
   "Valid GeneralizedTime CRL nextUpdate Test13",
   "Valid Negative Serial Number Test14",
   "Invalid Negative Serial Number Test15",
   "Valid Long Serial Number Test16",
   "Valid Long Serial Number Test17",
   "Invalid Long Serial Number Test18",
   "Valid Separate Certificate and CRL Keys Test19",
   "Invalid Separate Certificate and CRL Keys Test20",
   "Invalid Separate Certificate and CRL Keys Test21",
//"4.5 Verifying Paths with Self-Issued Certificates"
   "Valid Basic Self-Issued Old With New Test1",
   "Invalid Basic Self-Issued Old With New Test2",
   "Valid Basic Self-Issued New With Old Test3",
   "Valid Basic Self-Issued New With Old Test4",
   "Invalid Basic Self-Issued New With Old Test5",
   "Valid Basic Self-Issued CRL Signing Key Test6",
   "Invalid Basic Self-Issued CRL Signing Key Test7",
   "Invalid Basic Self-Issued CRL Signing Key Test8",
//"4.6 Verifying Basic Constraints"
   "Invalid Missing basicConstraints Test1",
   "Invalid cA False Test2",
   "Invalid cA False Test3",
   "Valid basicConstraints Not Critical Test4",
   "Invalid pathLenConstraint Test5",
   "Invalid pathLenConstraint Test6",
   "Valid pathLenConstraint Test7",
   "Valid pathLenConstraint Test8",
   "Invalid pathLenConstraint Test9",
   "Invalid pathLenConstraint Test10",
   "Invalid pathLenConstraint Test11",
   "Invalid pathLenConstraint Test12",
   "Valid pathLenConstraint Test13",
   "Valid pathLenConstraint Test14",
   "Valid Self-Issued pathLenConstraint Test15",
   "Invalid Self-Issued pathLenConstraint Test16",
   "Valid Self-Issued pathLenConstraint Test17",
//"4.7 Key Usage"
   "Invalid keyUsage Critical keyCertSign False Test1",
   "Invalid keyUsage Not Critical keyCertSign False Test2",
   "Valid keyUsage Not Critical Test3",
   "Invalid keyUsage Critical cRLSign False Test4",
   "Invalid keyUsage Not Critical cRLSign False Test5",
//"4.8 Certificate Policies"
   "All Certificates Same Policy Test1",
   "All Certificates No Policies Test2",
   "Different Policies Test3",
   "Different Policies Test4",
   "Different Policies Test5",
   "Overlapping Policies Test6",
   "Different Policies Test7",
   "Different Policies Test8",
   "Different Policies Test9",
   "All Certificates Same Policies Test10",
   "All Certificates AnyPolicy Test11",
   "Different Policies Test12",
   "All Certificates Same Policies Test13",
   "AnyPolicy Test14",
   "User Notice Qualifier Test15",
   "User Notice Qualifier Test16",
   "User Notice Qualifier Test17",
   "User Notice Qualifier Test18",
   "User Notice Qualifier Test19",
   "CPS Pointer Qualifier Test20",
//"4.9 Require Explicit Policy"
   "Valid RequireExplicitPolicy Test1",
   "Valid RequireExplicitPolicy Test2",
   "Invalid RequireExplicitPolicy Test3",
   "Valid RequireExplicitPolicy Test4",
   "Invalid RequireExplicitPolicy Test5",
   "Valid Self-Issued requireExplicitPolicy Test6",
   "Invalid Self-Issued requireExplicitPolicy Test7",
   "Invalid Self-Issued requireExplicitPolicy Test8",
//"4.10 Policy Mappings"
   "Valid Policy Mapping Test1",
   "Invalid Policy Mapping Test2",
   "Valid Policy Mapping Test3",
   "Invalid Policy Mapping Test4",
   "Valid Policy Mapping Test5",
   "Valid Policy Mapping Test6",
   "Invalid Mapping From anyPolicy Test7",
   "Invalid Mapping To anyPolicy Test8",
   "Valid Policy Mapping Test9",
   "Invalid Policy Mapping Test10",
   "Valid Policy Mapping Test11",
   "Valid Policy Mapping Test12",
   "Valid Policy Mapping Test13",
   "Valid Policy Mapping Test14",
//"4.11 Inhibit Policy Mapping"
   "Invalid inhibitPolicyMapping Test1",
   "Valid inhibitPolicyMapping Test2",
   "Invalid inhibitPolicyMapping Test3",
   "Valid inhibitPolicyMapping Test4",
   "Invalid inhibitPolicyMapping Test5",
   "Invalid inhibitPolicyMapping Test6",
   "Valid Self-Issued inhibitPolicyMapping Test7",
   "Invalid Self-Issued inhibitPolicyMapping Test8",
   "Invalid Self-Issued inhibitPolicyMapping Test9",
   "Invalid Self-Issued inhibitPolicyMapping Test10",
   "Invalid Self-Issued inhibitPolicyMapping Test11",
//"4.12 Inhibit Any Policy"
   "Invalid inhibitAnyPolicy Test1",
   "Valid inhibitAnyPolicy Test2",
   "inhibitAnyPolicy Test3",
   "Invalid inhibitAnyPolicy Test4",
   "Invalid inhibitAnyPolicy Test5",
   "Invalid inhibitAnyPolicy Test6",
   "Valid Self Issued inhibitAnyPolicy Test7",
   "Invalid Self Issued inhibitAnyPolicy Test8",
   "Valid Self Issued inhibitAnyPolicy Test9",
   "Invalid Self Issued inhibitAnyPolicy Test10",
//"4.13 Name Constraints"
   "Valid DN nameConstraints Test1",
   "Invalid DN nameConstraints Test2",
   "Invalid DN nameConstraints Test3",
   "Valid DN nameConstraints Test4",
   "Valid DN nameConstraints Test5",
   "Valid DN nameConstraints Test6",
   "Invalid DN nameConstraints Test7",
   "Invalid DN nameConstraints Test8",
   "Invalid DN nameConstraints Test9",
   "Invalid DN nameConstraints Test10",
   "Valid DN nameConstraints Test11",
   "Invalid DN nameConstraints Test12",
   "Invalid DN nameConstraints Test13",
   "Valid DN nameConstraints Test14",
   "Invalid DN nameConstraints Test15",
   "Invalid DN nameConstraints Test16",
   "Invalid DN nameConstraints Test17",
   "Valid DN nameConstraints Test18",
   "Valid Self-Issued DN nameConstraints Test19",
   "Invalid Self-Issued DN nameConstraints Test20",
   "Valid RFC822 nameConstraints Test21",
   "Invalid RFC822 nameConstraints Test22",
   "Valid RFC822 nameConstraints Test23",
   "Invalid RFC822 nameConstraints Test24",
   "Valid RFC822 nameConstraints Test25",
   "Invalid RFC822 nameConstraints Test26",
   "Valid DN and RFC822 nameConstraints Test27",
   "Invalid DN and RFC822 nameConstraints Test28",
   "Invalid DN and RFC822 nameConstraints Test29",
   "Valid DNS nameConstraints Test30",
   "Invalid DNS nameConstraints Test31",
   "Valid DNS nameConstraints Test32",
   "Invalid DNS nameConstraints Test33",
   "Valid URI nameConstraints Test34",
   "Invalid URI nameConstraints Test35",
   "Valid URI nameConstraints Test36",
   "Invalid URI nameConstraints Test37",
   "Invalid DNS nameConstraints Test38",
//"4.14 Distribution Points"
   "Valid distributionPoint Test1",
   "Invalid distributionPoint Test2",
   "Invalid distributionPoint Test3",
   "Valid distributionPoint Test4",
   "Valid distributionPoint Test5",
   "Invalid distributionPoint Test6",
   "Valid distributionPoint Test7",
   "Invalid distributionPoint Test8",
   "Invalid distributionPoint Test9",
   "Valid No issuingDistributionPoint Test10",
   "Invalid onlyContainsUserCerts CRL Test11",
   "Invalid onlyContainsCACerts CRL Test12",
   "Valid onlyContainsCACerts CRL Test13",
   "Invalid onlyContainsAttributeCerts Test14",
   "Invalid onlySomeReasons Test15",
   "Invalid onlySomeReasons Test16",
   "Invalid onlySomeReasons Test17",
   "Valid onlySomeReasons Test18",
   "Valid onlySomeReasons Test19",
   "Invalid onlySomeReasons Test20",
   "Invalid onlySomeReasons Test21",
   "Valid IDP with indirectCRL Test22",
   "Invalid IDP with indirectCRL Test23",
   "Valid IDP with indirectCRL Test24",
   "Valid IDP with indirectCRL Test25",
   "Invalid IDP with indirectCRL Test26",
   "Invalid cRLIssuer Test27",
   "Valid cRLIssuer Test28",
   "Valid cRLIssuer Test29",
   "Valid cRLIssuer Test30",
   "Invalid cRLIssuer Test31",
   "Invalid cRLIssuer Test32",
   "Valid cRLIssuer Test33",
   "Invalid cRLIssuer Test34",
   "Invalid cRLIssuer Test35",
//"4.15  Delta-CRLs"
   "Invalid deltaCRLIndicator No Base Test1",
   "Valid delta CRL Test2",
   "Invalid delta CRL Test3",
   "Invalid delta CRL Test4",
   "Valid delta CRL Test5",
   "Invalid delta CRL Test6",
   "Valid delta CRL Test7",
   "Valid delta CRL Test8",
   "Invalid delta CRL Test9",
   "Invalid delta CRL Test10",
//"4.16  Private Certificate Extensions"
   "Valid Unknown Not Critical Certificate Extension Test1",
   "Invalid Unknown Critical Certificate Extension Test2"
 };



class NIST_PKITSTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(NIST_PKITSTest);
  CPPUNIT_TEST(Test_All);
  CPPUNIT_TEST_SUITE_END();

public:
  NIST_PKITSTest() : cert_dir(BASE_LOCATION + "certs"), crl_dir(BASE_LOCATION + "crls"), trusted_cert_dir(BASE_LOCATION + "certificates") {}; 
  void setUp();
  void tearDown() {};
  void Test_All();

private:
  bool changeCACertsandCRLsName(void);
  bool doValidation(const std::string& cert_file, AuthN::Validator::ValidationMode mode, bool valid = true);

private:
  std::string cert_dir;
  std::string crl_dir;
  std::string trusted_cert_dir; //This will contain ca and crl files with hash value as name
  int count;
};

static bool copy_cacert_files(const std::string& source, const std::string& destination) {
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  DIR* dir;
  //Handle the CA files
  dir = opendir(source.c_str());
  if(!dir) { return false; };
  for(;;) {
    struct dirent* d = ::readdir(dir);
    if(!d) break;
    if(strcmp(d->d_name,".") == 0) continue;
    if(strcmp(d->d_name,"..") == 0) continue;
    std::string file = source + "/" + d->d_name;
   
    struct stat st;
    if(stat(file.c_str(),&st) != 0) {
      std::cout<<"Failed to find the file: "<<file<<std::endl; continue;
    }
    if(!S_ISREG(st.st_mode)) {
      std::cout<<"The location: "<<file<<" is not a regular file"<<std::endl; continue;
    }

    AuthN::Credentials cred(ctx);
    std::ifstream cert_stream, key_stream, chain_stream;
    cert_stream.open(file.c_str(), std::ios::in | std::ios::binary);
    cred.Assign(cert_stream, chain_stream, key_stream);
    cert_stream.close();
    if((cred.GetCA() == false) && (file.find("CRLSigning") == std::string::npos)) continue;
    //If it is not for signing CA and signing CRL


    std::string cert_str;
    cred.GetCertificate(cert_str);
    CPPUNIT_ASSERT(!cert_str.empty());
    AuthN::OpenSSL::CertContext cert_ctx(ctx, cert_str);
    X509* cert = cert_ctx.getCert();
    CPPUNIT_ASSERT(cert!=NULL);
    std::string hash_str = AuthN::OpenSSL::get_cert_hash(cert);

    std::string dest = destination;
    std::string dest_file;
    dest_file = dest +  "/" + hash_str + ".0";
    if(AuthN::Utils::File::file_exist(dest, dest_file)) {
      dest_file = dest + "/" + hash_str + ".1";
    }

    std::ofstream out(dest_file.c_str(), std::ios::out);
    cred.GetCertificate(out);
    out.close();
  }
  closedir(dir);
  return true;
}

static bool copy_crl_files(const std::string& source, const std::string& destination) {
  DIR* dir;
  //Handle the CRL files
  dir = opendir(source.c_str());
  if(!dir) { return false; };
  for(;;) {
    struct dirent* d = ::readdir(dir);
    if(!d) break;
    if(strcmp(d->d_name,".") == 0) continue;
    if(strcmp(d->d_name,"..") == 0) continue;
    std::string file = source + "/" + d->d_name;
   
    struct stat st;
    if(stat(file.c_str(),&st) != 0) {
      std::cout<<"Failed to find the file: "<<file<<std::endl; continue;
    }
    if(!S_ISREG(st.st_mode)) {
      std::cout<<"The location: "<<file<<" is not a regular file"<<std::endl; continue;
    }

    AuthN::Context context(AuthN::Context::EmptyContext);

    X509_CRL* crl = NULL;
    //AuthN::OpenSSL::loadCRLFile(context, file, crl, false);
    AuthN::OpenSSL::CRLContext crl_ctx(&context, file.c_str());
    crl = crl_ctx.getCRL();
    CPPUNIT_ASSERT(crl != NULL);

    std::string hash_str;
    if(crl) hash_str = AuthN::OpenSSL::get_crl_hash(crl);

    std::string dest = destination;
    std::string dest_file;
    dest_file = dest +  "/" + hash_str + ".r0";
    if(AuthN::Utils::File::file_exist(dest, dest_file)) {
      dest_file = dest + "/" + hash_str + ".r1";
    }

    BIO* out=BIO_new(BIO_s_file());
    CPPUNIT_ASSERT(out != NULL);
    int i = BIO_write_filename(out, (void*)(dest_file.c_str()));
    CPPUNIT_ASSERT(i>0);
    i = PEM_write_bio_X509_CRL(out, crl);
    CPPUNIT_ASSERT(i != 0);
    BIO_free_all(out);

    //if(crl) X509_CRL_free(crl);
  }
  closedir(dir);
  return true;
}


bool NIST_PKITSTest::changeCACertsandCRLsName() {
  bool ret;
  ret = copy_cacert_files(cert_dir, trusted_cert_dir);

  CPPUNIT_ASSERT(ret);
  ret = copy_crl_files(crl_dir, trusted_cert_dir);
  CPPUNIT_ASSERT(ret);
  return true;
}

void NIST_PKITSTest::setUp() {
  CPPUNIT_ASSERT(!cert_dir.empty());
  CPPUNIT_ASSERT(!crl_dir.empty());
  CPPUNIT_ASSERT(!trusted_cert_dir.empty());
  AuthN::Utils::File::removedir(trusted_cert_dir);
  CPPUNIT_ASSERT(!AuthN::Utils::File::makedir(trusted_cert_dir.c_str()));
  changeCACertsandCRLsName();
  count = 0;

}

bool NIST_PKITSTest::doValidation(const std::string& cert_file, AuthN::Validator::ValidationMode mode, bool valid) {
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  ctx.SetCAPath(trusted_cert_dir);
  ctx.SetCRLPath(trusted_cert_dir);

  AuthN::Credentials cred(ctx);
  std::ifstream cert_stream, key_stream, chain_stream;
  cert_stream.open(cert_file.c_str(), std::ios::in | std::ios::binary);
  cred.Assign(cert_stream, chain_stream, key_stream);
  cert_stream.close();

  AuthN::Validator validator(ctx);
  validator.SetMode(mode);
  cred.SetValidator(validator);

  AuthN::Status stat;
  stat = cred.Validate();
  if(valid) CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  else CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1));

  return true;
}

static void remove_space(std::string& str) {
  std::string tmp;
  std::size_t pos1, pos2;
  while (1) {
    pos1 = str.find(" ");
    if(pos1 != std::string::npos ) {
      pos2 = str.find_first_not_of(" ", pos1);
      tmp = str.substr(0, pos1);
      if(pos2 != std::string::npos) tmp.append(str.substr(pos2));
      str = tmp;
    }
    else break;
  }
}

void NIST_PKITSTest::Test_All() {
  AuthN::Validator::ValidationMode mode = AuthN::Validator::ValidationCRLMandatory;
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  int num = sizeof(cert_file_names)/sizeof(char*);
  for(int i=0; i<num; i++) {
    std::string file_name = *(cert_file_names+i);
    remove_space(file_name);
    file_name.append("EE.crt");
    bool valid;
    std::string file_loc;
    file_loc = cert_dir + "/" + file_name;
std::cout<<"cert dir: "<<cert_dir<<std::endl;
    if(AuthN::Utils::File::file_exist(file_loc)) {
      if(file_loc.find("Invalid") != std::string::npos) valid = false;
      else valid = true;
    }
    else  { 
      std::cout<<"cert file: "<<file_loc<<" does not exist"<<std::endl;
      continue;
    }

    std::cout<<std::endl;
    std::cout<<"====== Test Begins ======"<<std::endl;

    AuthN::Credentials cred(ctx);
    std::ifstream cert_stream, key_stream, chain_stream;
    cert_stream.open(file_loc.c_str(), std::ios::in | std::ios::binary);
    cred.Assign(cert_stream, chain_stream, key_stream);
    cert_stream.close();
    if(cred.GetCA() == true) continue;  //Only handle the non-CA files

    doValidation(file_loc, mode, valid);

    count++;
    std::cout<<"The "<<count<<" test passed"<<std::endl;
    std::cout<<"====== Test Ends ======"<<std::endl;
  }

}

CPPUNIT_TEST_SUITE_REGISTRATION(NIST_PKITSTest);

