function scatter_plots(X,xp,vnames,plotsymbol, fnam, dirname, figtitle, xparam1, options_)
% scatter_plots(X,xp,vnames,plotsymbol, fnam, dirname, figtitle, xparam1, options_)
% Pairwise scatter plots of the columns of x and y after Monte Carlo filtering
% Inputs:
%  - X              [double]    nxk matrix with columns containing behavioural sample
%  - xp             [double]    mxk matrix with columns containing non-behavioural sample
%  - vnames         [char]      vector of variable names (default = numeric labels 1,2,3 etc.)
%  - plotsymbol     [char]      plt symbol (default = '.' for npts > 100, 'o' for npts < 100
%  - fnam           [char]      figure name
%  - dirname        [char]      directory name
%  - figtitle       [char]      figure title
%  - xparam1        [double]    parameter vector
%  - options_       [struct]    option structure
%
% Written by Marco Ratto
% Joint Research Centre, The European Commission,
% marco.ratto@ec.europa.eu


% Copyright © 2017 European Commission
% Copyright © 2017-2023 Dynare Team
%
% This file is part of Dynare.
%
% Dynare is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Dynare is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Dynare.  If not, see <https://www.gnu.org/licenses/>.

[n,p] = size(X);

nflag = 0;
if nargin >=3
    nflag = 1;
end

if nargin<4 || isempty(plotsymbol)
    if n*p<100
        plotsymbol = 'o';
    else 
        plotsymbol = '.';
    end
end

if nargin<5 || isempty(fnam)
    fnam='scatter_plot';
end
if nargin<6 || isempty(dirname)
    dirname='';
    nograph=1;
    options_.nodisplay=0;
else
    nograph=0;
end
if nargin<7 || isempty(figtitle)
    figtitle=fnam;
end
if nargin<8
    xparam1=[];
end

figtitle_tex=strrep(figtitle,'_','\_');

fig_nam_=fnam;

hh_fig=dyn_figure(options_.nodisplay,'name',figtitle);
set(hh_fig,'userdata',{X,xp})

bf = 0.1;
ffs = 0.05/(p-1);
ffl = (1-2*bf-0.05)/p;
if p>1
    fL = linspace(bf,1-bf+ffs,p+1);
else
    fL = bf;
end
for i = 1:p
    for j = 1:p
        h = axes('position',[fL(i),fL(p+1-j),ffl,ffl]);
        if i==j
            h1=gsa.cumplot(X(:,j));
            set(h,'Tag','cumplot')
            set(h1,'color',[0 0 1],'LineWidth',1.5)
            if ~isempty(xparam1)
                hold on, plot(xparam1([j j]),[0 1],'k--')
            end
            if j<p
                set(gca,'XTickLabel',[],'XTick',[]);
            else
                grid off
            end
            set(gca,'YTickLabel',[],'YTick',[]);
        else
            plot(X(:,i),X(:,j),[plotsymbol,'b'])
            set(h,'Tag','scatter')

            if ~isempty(xparam1)
                hold on, plot(xparam1(i),xparam1(j),'s','MarkerFaceColor',[0 0.75 0],'MarkerEdgeColor',[0 0.75 0])
            end
            hold off;
            if i<p
                set(gca,'YTickLabel',[],'YTick',[]);
            else
                set(gca,'yaxislocation','right');
            end
            if j<p
                set(gca,'XTickLabel',[],'XTick',[]);
            end
        end
        if nflag == 1
            set(gca,'fontsize',9);
        end
        if i==1
            if nflag == 1
                if options_.TeX
                    ylabel(vnames(j,:),'Rotation',45,'interpreter','latex', ...
                        'HorizontalAlignment','right','VerticalAlignment','middle');
                else
                    ylabel(vnames(j,:),'Rotation',45,'interpreter','none', ...
                        'HorizontalAlignment','right','VerticalAlignment','middle');
                end
            else
                ylabel([num2str(j),' '],'Rotation',90)
            end
        end
        if j==1
            if nflag == 1
                if options_.TeX
                    title(vnames(i,:),'interpreter','latex','Rotation',45, ...
                          'HorizontalAlignment','left','VerticalAlignment','bottom')
                else
                    title(vnames(i,:),'interpreter','none','Rotation',45, ...
                          'HorizontalAlignment','left','VerticalAlignment','bottom')            
                end
            else
                title(num2str(i))
            end
        end
        drawnow
    end
end

if ~nograph
    dyn_saveas(hh_fig,[dirname,filesep,fig_nam_],options_.nodisplay,options_.graph_format);
    if options_.TeX && any(strcmp('eps',cellstr(options_.graph_format)))
        fidTeX = fopen([dirname,'/',fig_nam_ '.tex'],'w');
        fprintf(fidTeX,'%% TeX eps-loader file generated by scatter_plots.m (Dynare).\n');
        fprintf(fidTeX,['%% ' datestr(now,0) '\n\n']);
        fprintf(fidTeX,'\\begin{figure}[H]\n');
        fprintf(fidTeX,'\\centering \n');
        fprintf(fidTeX,'\\includegraphics[width=0.8\\textwidth]{%s}\n',strrep([dirname,'/',fig_nam_],'\','/'));
        fprintf(fidTeX,'\\caption{%s.}',figtitle_tex);
        fprintf(fidTeX,'\\label{Fig:%s}\n',fig_nam_);
        fprintf(fidTeX,'\\end{figure}\n\n');
        fprintf(fidTeX,'%% End Of TeX file. \n');
        fclose(fidTeX);
    end
end