package tests::AggregateTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::ReportSpec;
use Lire::Aggregate;
use tests::MockAggregator;

use Class::Inner;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'test-aggregate' );

    $self->{'mock_aggregator'} =
      new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'mock_aggr'} =
      new Class::Inner( 'parent' => 'Lire::Aggregate',
                        'args' => [ 'report_spec' => $self->{'spec'},
                                    'name' => 'aggrTest',
                                    'parent' => $self->{'mock_aggregator'},
                                    'op' => 'mock',
                                  ],
                        'methods' =>
                        {
                         'new' => sub {
                             my $self = bless {}, shift;
                             $self->init( @_ );
                             return $self;
                         },
                         'sql_required_fields' => sub {
                             return [ 'time-taken', 'connection_id' ]
                         },
                         'sql_aggr_expr' => sub {
                             return 'lr_mock_aggr(time_taken)';
                         },
                        }
                      );

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down();

    return;
}

sub test_init {
    my $self = $_[0];

    my $aggr = $self->{'mock_aggr'};
    my %args = ( 'report_spec' => $self->{'spec'},
                 'name' => 'aggrTest',
                 'op' => 'mock' );
    $self->assert_died( sub { $aggr->init( %args ) },
                        qr/missing 'parent' parameter/ );
    $args{'parent'} = $self->{'mock_aggregator'};
    delete $args{'name'};
    $self->assert_died( sub { $aggr->init( %args ) },
                        qr/missing 'name' parameter/ );

    $self->assert_str_equals( 'aggrTest', $aggr->{'name'} );
    $self->assert_str_equals( 'mock', $aggr->{'op'} );
    $self->assert_str_equals( $self->{'spec'}, $aggr->{'report_spec'} );
}

sub test_build_query {
    my $self = $_[0];

    my $e_query = new Lire::DlfQuery( 'test' );
    $e_query->add_aggr_field( 'aggrTest', 'lr_mock_aggr(time_taken)' );
    $e_query->add_aggr_field( '_lr_aggrTest_mc',
                              'lr_missing_cases("time-taken",connection_id)' );
    my $query = new Lire::DlfQuery( 'test' );
    $self->{'mock_aggr'}->build_query( $query );

    $self->assert_deep_equals( $e_query, $query );
}

sub test_set_missing_cases_aggr_expr {
    my $self = $_[0];

    my $e_query = new Lire::DlfQuery( 'test' );
    $e_query->add_aggr_field( '_lr_aggrTest_mc',
                              'lr_missing_cases("time-taken",connection_id)' );

    my $query = new Lire::DlfQuery( 'test' );
    $self->{'mock_aggr'}->set_missing_cases_aggr_expr( $query );

    $self->assert_deep_equals( $e_query, $query );
}

sub test_set_missing_cases_value {
    my $self = $_[0];

    my $value = {};
    $self->{'mock_aggr'}->set_missing_cases_value( { '_lr_aggrTest_mc' => 10 },
                                                   $value );
    $self->assert_equals( 10, $value->{'missing_cases'} );
}

sub test_set_name {
    my $self = $_[0];

    $self->{'mock_aggr'}->set_name( 'newName' );
    $self->assert_str_equals( 'newName', $self->{'mock_aggr'}{'name'} );

    $self->assert_died( sub { $self->{'mock_aggr'}->set_name( undef ) },
                        qr/missing 'name' parameter/ );

    $self->assert_died( sub { $self->{'mock_aggr'}->set_name( 'invalid name' ) },
                        qr/invalid field name: 'invalid name'/ );

    $self->{'mock_aggregator'}{'name'} = 'aTest';
    $self->assert_died( sub { $self->{'mock_aggr'}->set_name( 'aTest' ) },
                        qr/name 'aTest' is already defined/ );

    # Check that we can call set_name() with the current value
    $self->{'mock_aggr'}->set_name( 'newName' );
}

1;
