/*
 * Copyright (c) 2015, 2019, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/*
 * @test
 * @summary Test to see if thread interrupt handling interferes with other threads.
 * @run main/othervm -Djdk.image.map.all=false JImageOpenTest
 */
public class JImageOpenTest {
    private static final int NTHREADS = 10;

    public static void main(String[] args) throws Exception {

        final FileSystem fs = FileSystems.getFileSystem(URI.create("jrt:/"));
        final Path root = fs.getPath("/modules");

        final List<String> names = Files.walk(root)
                .filter(p -> p.getNameCount() > 2)
                .filter(p -> ModuleLayer.boot().findModule(p.getName(1).toString()).isPresent())
                .map(p -> p.subpath(2, p.getNameCount()))
                .map(p -> p.toString())
                .filter(s -> s.endsWith(".class") && !s.endsWith("module-info.class"))
                .collect(Collectors.toList());

        Runnable r = new Runnable() {
            @Override
            public void run() {
                names.forEach(name -> {
                    String cn = name.substring(0, name.length() - 6).replace('/', '.');
                    try {
                        Class.forName(cn, false, ClassLoader.getSystemClassLoader());
                    } catch (Exception ex) {
                        System.err.println(Thread.currentThread() + " " + ex.getClass());
                    }
                });
            }
        };

        Thread[] threads = new Thread[NTHREADS];

        for (int i = 0; i < NTHREADS; i++) {
            Thread thread =  new Thread(r);
            threads[i] = thread;
            thread.start();
        }

        Thread.sleep(1);

        for (int i = 0; i < NTHREADS; i++) {
            Thread thread = threads[i];

            if (thread.isAlive()) {
                thread.interrupt();
                break;
            }
        }

        for (int i = 0; i < NTHREADS; i++) {
            Thread thread = threads[i];
            thread.join();
        }
    }
}
