<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exslt="http://exslt.org/common"
                xmlns:str="http://exslt.org/strings"
                extension-element-prefixes="exslt str"
                version="1.0">

<!-- 
     This file contains templates for upgrading PhotoML files from
     DTD version 0.11 to version 0.12.

     Copyright © 2007 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->

<xsl:output method="xml" indent="yes" encoding="utf-8"
	    doctype-public="-//BW//DTD PhotoML 0.12//EN"
            doctype-system="photo.dtd"/>


<!-- Top level template controlling two-pass processing -->
<xsl:template match="/">
  <xsl:variable name="pass1">
    <xsl:copy>
      <xsl:apply-templates select="@* | node()"/>
    </xsl:copy>
  </xsl:variable>
  <xsl:apply-templates select="exslt:node-set($pass1)/*" mode="pass2"/>
</xsl:template>


<!-- Default template -->
<xsl:template match="@* | node()">
   <xsl:copy>
     <xsl:apply-templates select="@* | node()"/>
   </xsl:copy>
</xsl:template>


<!-- Handle replacement of developer/type by developer/{make,name} -->
<xsl:template match="type[parent::developer]">
  <xsl:choose>
    <xsl:when test=". = 'DD-X'">
      <make>Ilford</make>
      <name>DD-X</name>
    </xsl:when>
    <xsl:when test=". = 'HC-110'">
      <make>Kodak</make>
      <name>HC-110</name>
    </xsl:when>
    <xsl:when test=". = 'Ilfotec DD-X'">
      <make>Ilfotec</make>
      <name>DD-X</name>
    </xsl:when>
    <xsl:when test=". = 'Rodinal'">
      <make>Agfa</make>
      <name>Rodinal</name>
    </xsl:when>
    <xsl:otherwise>
      <xsl:comment> Upgrade warning: editing might be required to correctly represent the previous type element across the new make and name elements. </xsl:comment>
      <xsl:message>
        <xsl:text>Warning: manual editing of output may be required (position commented)
         Element developer/type replaced by developer/{make,name}</xsl:text>
      </xsl:message>
      <name><xsl:value-of select="."/></name>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>


<!-- Handle change of semantics of frame/size -->
<xsl:template match="size[parent::frame]">
  <xsl:comment> Upgrade warning: editing might be required to conform with the new format of the frame/size element. </xsl:comment>
  <xsl:message>
    <xsl:text>Warning: manual editing of output may be required (position commented)
         Format of frame/size has changed</xsl:text>
  </xsl:message>
  <size>
    <xsl:apply-templates select="@*"/>
    <xsl:value-of select="."/>
  </size>
</xsl:template>


<!-- Handle renaming of scene/type to scene/genre -->
<xsl:template match="type[parent::scene]">
  <genre><xsl:value-of select="."/></genre>
</xsl:template>


<!-- Handle change of order of children of frame/scene -->
<xsl:template match="scene[parent::frame]">
  <scene>
    <xsl:apply-templates select="occasion | 
                       node()[following-sibling::occasion and not(self::*) and
                       not(following-sibling::*[1][not(self::occasion)])]"/>
    <xsl:apply-templates select="location | 
                       node()[following-sibling::location and not(self::*) and
                       not(following-sibling::*[1][not(self::location)])]"/>
    <xsl:apply-templates select="description | 
                    node()[following-sibling::description and not(self::*) and
                    not(following-sibling::*[1][not(self::description)])]"/>
    <xsl:apply-templates select="caption | 
                       node()[following-sibling::caption and not(self::*) and
                       not(following-sibling::*[1][not(self::caption)])]"/>
    <xsl:apply-templates select="type | 
                       node()[following-sibling::type and not(self::*) and
                       not(following-sibling::*[1][not(self::type)])]"/>
    <xsl:apply-templates select="keywords | 
                       node()[following-sibling::keywords and not(self::*) and
                       not(following-sibling::*[1][not(self::keywords)])]"/>
    <xsl:apply-templates select="content | 
                       node()[following-sibling::content and not(self::*) and
                       not(following-sibling::*[1][not(self::content)])]"/>
    <xsl:apply-templates select="note | 
                       node()[following-sibling::note and not(self::*) and
                       not(following-sibling::*[1][not(self::note)])]"/>
    <xsl:apply-templates select="node()[not(self::*) and 
                                        not(following-sibling::*)]"/>
  </scene>
</xsl:template>


<!-- Handle removal of frame/viewpoint -->
<xsl:template match="viewpoint[parent::frame]">
  <xsl:comment>
    <xsl:text> Upgrade warning: frame/viewpoint has been removed - content should be manually transferred to frame/scene/location.  </xsl:text>
    <xsl:text> &lt;viewpoint&gt;</xsl:text>
    <xsl:apply-templates select="@* | node()" mode="treeastext"/>
    <xsl:text>&lt;/viewpoint&gt; </xsl:text>
  </xsl:comment>
  <xsl:message>
    <xsl:text>Warning: manual editing of output required (position commented)
         Element frame/viewpoint has been removed</xsl:text>
  </xsl:message>
</xsl:template>


<!-- Handle change of object/type to object/@type -->
<xsl:template match="object[child::type]">
  <object type='{type}'>
    <xsl:apply-templates select="@*"/>
    <xsl:apply-templates select="*[name() != 'type'] |
                                 text()[not(following-sibling::type)] | 
                                 comment() | processing-instruction()"/>
  </object>
</xsl:template>


<!-- Handle change of device/type to device/@type -->
<xsl:template match="device[child::type]">
  <device type='{type}'>
    <xsl:apply-templates select="@*"/>
    <xsl:apply-templates select="*[name() != 'type'] |
                                 text()[not(following-sibling::type)] | 
                                 comment() | processing-instruction()"/>
  </device>
</xsl:template>


<!-- Handle change of order of children of equipment/body -->
<xsl:template match="body[parent::equipment]">
  <body>
    <xsl:apply-templates select="make | 
                       node()[following-sibling::make and not(self::*) and
                       not(following-sibling::*[1][not(self::make)])]"/>
    <xsl:apply-templates select="model | 
                       node()[following-sibling::model and not(self::*) and
                       not(following-sibling::*[1][not(self::model)])]"/>
    <xsl:apply-templates select="serial | 
                    node()[following-sibling::serial and not(self::*) and
                    not(following-sibling::*[1][not(self::serial)])]"/>
    <xsl:apply-templates select="type | 
                       node()[following-sibling::type and not(self::*) and
                       not(following-sibling::*[1][not(self::type)])]"/>
    <xsl:apply-templates select="note | 
                       node()[following-sibling::note and not(self::*) and
                       not(following-sibling::*[1][not(self::note)])]"/>
    <xsl:apply-templates select="node()[not(self::*) and 
                                        not(following-sibling::*)]"/>
  </body>
</xsl:template>


<!-- Handle change of order of children of equipment/lens -->
<xsl:template match="lens[parent::equipment]">
  <lens>
    <xsl:apply-templates select="make | 
                       node()[following-sibling::make and not(self::*) and
                       not(following-sibling::*[1][not(self::make)])]"/>
    <xsl:apply-templates select="model | 
                       node()[following-sibling::model and not(self::*) and
                       not(following-sibling::*[1][not(self::model)])]"/>
    <xsl:apply-templates select="serial | 
                    node()[following-sibling::serial and not(self::*) and
                    not(following-sibling::*[1][not(self::serial)])]"/>
    <xsl:apply-templates select="type | 
                       node()[following-sibling::type and not(self::*) and
                       not(following-sibling::*[1][not(self::type)])]"/>
    <xsl:apply-templates select="note | 
                       node()[following-sibling::note and not(self::*) and
                       not(following-sibling::*[1][not(self::note)])]"/>
    <xsl:apply-templates select="node()[not(self::*) and 
                                        not(following-sibling::*)]"/>
  </lens>
</xsl:template>


<!-- Handle splitting of date elements into date and time -->
<xsl:template match="date">
  <xsl:variable name="dttkn" select="str:tokenize(.,'T')"/>
  <xsl:variable name="d" select="string($dttkn[1])"/>
  <xsl:variable name="t" select="string($dttkn[2])"/>
  <date><xsl:value-of select="$d"/></date>
  <xsl:if test="$t">
    <xsl:variable name="tztkn" select="str:tokenize($t,'+-Z')"/>
    <xsl:variable name="tt" select="string($tztkn[1])"/>
    <xsl:variable name="tz" select="string($tztkn[2])"/>
    <time>
      <xsl:if test="$tz">
        <xsl:attribute name="zone">
          <xsl:choose>
            <xsl:when test="contains($t,'-')">
                <xsl:value-of select="concat('-',$tz)"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="concat('+',$tz)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:attribute>
      </xsl:if>
      <xsl:value-of select="$tt"/>
    </time>
  </xsl:if>
</xsl:template>


<!-- Handle renaming of location/suburb to location/subcity -->
<xsl:template match="suburb[parent::location]">
  <subcity><xsl:value-of select="."/></subcity>
</xsl:template>


<!-- Handle renaming of location/street to location/road -->
<xsl:template match="street[parent::location]">
  <road><xsl:value-of select="."/></road>
</xsl:template>


<!-- Handle removal of location/annotation -->
<xsl:template match="annotation[parent::location]">
  <xsl:comment>
    <xsl:text> Upgrade warning: location/annotation has been removed - content should be manually transferred to siblings.  </xsl:text>
    <xsl:text> &lt;annotation&gt;</xsl:text>
    <xsl:value-of select="."/>
    <xsl:text>&lt;/annotation&gt; </xsl:text>
  </xsl:comment>
  <xsl:message>
    <xsl:text>Warning: manual editing of output required (position commented)
         Element location/annotation has been removed</xsl:text>
  </xsl:message>
</xsl:template>


<!-- Handle renaming of hardware/manufacturer to hardware/make -->
<xsl:template match="manufacturer[parent::hardware]">
  <make><xsl:value-of select="."/></make>
</xsl:template>


<!-- Handle renaming of properties/bits to properties/bit-depth -->
<xsl:template match="bits[parent::properties]">
  <bit-depth><xsl:value-of select="."/></bit-depth>
</xsl:template>


<!-- Handle renaming of properties/fileformat to properties/file-format -->
<xsl:template match="fileformat[parent::properties]">
  <file-format>
    <xsl:apply-templates select="@*"/>
    <xsl:value-of select="."/>
  </file-format>
</xsl:template>


<!-- Handle renaming of properties/iccprofile to properties/icc-profile -->
<xsl:template match="iccprofile[parent::properties]">
  <icc-profile>
    <xsl:apply-templates select="@*"/>
    <xsl:value-of select="."/>
  </icc-profile>
</xsl:template>


<!-- Handle change in location content model, which previously
     disallowed joint occurrence of street (now road), place, and
     description children -->
<xsl:template match="location[parent::scene]
                       [(child::street and 
                           (preceding::defaults[not(parent::frame-set)] | 
                            ancestor::frame-set/defaults)
                              [descendant::location/place or 
                               descendant::location/description]) or
                        (child::place and 
                            (preceding::defaults[not(parent::frame-set)] | 
                            ancestor::frame-set/defaults)
                              [descendant::location/street or 
                               descendant::location/description]) or
                        (child::description and 
                            (preceding::defaults[not(parent::frame-set)] | 
                            ancestor::frame-set/defaults)
                              [descendant::location/street or
                               descendant::location/place])]">
  <xsl:message>
    <xsl:text>Warning: manual editing of output may be required
         Element location content model has changed</xsl:text>
  </xsl:message>
  <xsl:copy>
    <xsl:value-of select="text()[1]"/>
    <xsl:choose>
      <xsl:when test="child::street">
        <xsl:processing-instruction name='merge:reject'>
          <xsl:text>place description</xsl:text>
        </xsl:processing-instruction>
      </xsl:when>
      <xsl:when test="child::place">
        <xsl:processing-instruction name='merge:reject'>
          <xsl:text>road description</xsl:text>
        </xsl:processing-instruction>
      </xsl:when>
      <xsl:when test="child::description">
        <xsl:processing-instruction name='merge:reject'>
          <xsl:text>road place</xsl:text>
        </xsl:processing-instruction>
      </xsl:when>
    </xsl:choose>
     <xsl:apply-templates select="@* | node()"/>
   </xsl:copy>
</xsl:template>


<!-- Default template for mode 'pass2' -->
<xsl:template match="@* | node()" mode="pass2">
   <xsl:copy>
     <xsl:apply-templates select="@*" mode="pass2"/>
     <xsl:apply-templates select="node()" mode="pass2"/>
   </xsl:copy>
</xsl:template>


<!-- Handle change in defaults merge prune signal from empty element
    to processing instruction. --> 
<xsl:template match="*[not(child::*) and not(@*) and .='' and
                     preceding::*[ancestor::defaults and
                     name(current())=name(.) and name(current()/..)=name(..)]]"
                     mode="pass2">
  <xsl:processing-instruction name='merge:reject'>
    <xsl:value-of select="name(.)"/>
  </xsl:processing-instruction>
</xsl:template>


<!-- Handle change in merge processing of attributes of text-only elements. -->
<xsl:template match="*[not(child::*) and .!='' and 
                     preceding::*[ancestor::defaults and @* and 
                     name(current())=name(.) and name(current()/..)=name(..)]]"
                     mode="pass2">
  <xsl:variable name="pnds" select="."/>
  <xsl:variable name="ppth" select="concat(name(./..),'/',name(.))"/>
  <xsl:variable name="attlst">
    <xsl:for-each select="preceding::*[ancestor::defaults and @* and 
                          concat(name(./..),'/',name(.))=$ppth][1]/@*">
      <xsl:if test="not($pnds/@*[name(current()) = name(.)])">
        <xsl:value-of select="concat('@',name(.),' ')"/>
      </xsl:if>
    </xsl:for-each>
  </xsl:variable>
  <xsl:copy>
    <xsl:apply-templates select="@*" mode="pass2"/>
    <xsl:if test="normalize-space($attlst) != ''">
      <xsl:comment> Upgrade warning: defaults expansion involving text content elements with attributes may give different results under the new version - please confirm that the merge:reject processing instruction inserted here gives the desired result. </xsl:comment>
      <xsl:message>
        <xsl:text>Warning: manual editing of output may be required (position commented)
         Handling of attributes in defaults expansion of text content
         elements has changed</xsl:text>
      </xsl:message>
      <xsl:processing-instruction name='merge:reject'>
        <xsl:value-of select="normalize-space($attlst)"/>
      </xsl:processing-instruction>
    </xsl:if>
    <xsl:apply-templates select="node()" mode="pass2"/>
  </xsl:copy>
</xsl:template>


<!-- Handle display of elements as text within commented-out sections -->
<xsl:template match="*" mode="treeastext">
  <xsl:value-of select="concat('&lt;',name())"/>
  <xsl:apply-templates select="@*" mode="treeastext"/>
  <xsl:text>&gt;</xsl:text>
  <xsl:apply-templates select="node()" mode="treeastext"/>
  <xsl:value-of select="concat('&lt;/',name(),'&gt;')"/>
</xsl:template>


<!-- Handle display of attributes as text within commented-out sections -->
<xsl:template match="@*" mode="treeastext">
  <xsl:value-of select="concat(' ',name(),'=','&quot;',.,'&quot;')"/>
</xsl:template>


<!-- Handle display of comments as text within commented-out sections -->
<xsl:template match="comment()" mode="treeastext">
  <xsl:value-of select="concat('&lt;!COMMENT',.,'COMMENT&gt;')"/>
</xsl:template>


<!-- Handle display of processing instructions as text within
     commented-out sections --> 
<xsl:template match="processing-instruction()" mode="treeastext">
  <xsl:value-of select="concat('&lt;?',name(),' ',.,'?&gt;')"/>
</xsl:template>


<!-- Handle display of text nodes within commented-out sections -->
<xsl:template match="text()" mode="treeastext">
  <xsl:value-of select="."/>
</xsl:template>


</xsl:stylesheet>
