/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifndef PERCENTILES_HIST_H_
#define PERCENTILES_HIST_H_

#include <assert.h>

#include "field.h"

struct Histogram
{
  double min;
  double max;
  double step;
  int nbins;
  int nsamp;
  void *ptr;
};

class HistogramSet
{
private:
  int nvars;
  std::vector<int> var_nlevels;
  std::vector<size_t> var_nhists;
  std::vector<std::vector<std::vector<Histogram>>> histograms;

  void
  init()
  {
    var_nlevels.resize(nvars, 0);
    var_nhists.resize(nvars, 0);
    histograms.resize(nvars);
  }

public:
  HistogramSet() { nvars = 0; }

  HistogramSet(int _nvars) : nvars(_nvars)
  {
    assert(nvars > 0);
    init();
  }

  void
  create(int _nvars)
  {
    nvars = _nvars;
    assert(nvars > 0);
    init();
  }

  ~HistogramSet()
  {
    for (int varID = nvars; varID-- > 0;)
      {
        size_t nhists = this->var_nhists[varID];
        for (int levelID = this->var_nlevels[varID]; levelID-- > 0;)
          {
            for (size_t histID = nhists; histID-- > 0;) free(this->histograms[varID][levelID][histID].ptr);
          }
      }
  }

  void createVarLevels(int varID, int nlevels, size_t nhists);
  void defVarLevelBounds(int varID, int levelID, const Field &field1, const Field &field2);
  void defVarLevelBoundsF(int varID, int levelID, const Field &field1, const Field &field2);
  int addSubVarLevelValues(int varID, int levelID, const Field &field, int operation, int ptype);
  void getVarLevelPercentiles(Field &field, int varID, int levelID, double p, int ptype);
  void Reset(int varID, int levelID, int ptype);
};

#endif /* PERCENTILES_HIST_H_ */
